// Photo Courses Website - Version 3 (HTML/CSS/JS)

const courseLevels = [
  { 
    id: 'beginner', 
    name: 'Начинающий', 
    icon: 'icons/level-beginner.svg',
    color: '#8B5CF6',
    description: 'Основы фотографии для новичков',
    gradient: 'linear-gradient(135deg, #8B5CF6 0%, #EC4899 100%)',
    duration: '8 недель',
    lessons: '24 урока'
  },
  { 
    id: 'advanced', 
    name: 'Продвинутый', 
    icon: 'icons/level-advanced.svg',
    color: '#EC4899',
    description: 'Углубленное изучение техник',
    gradient: 'linear-gradient(135deg, #EC4899 0%, #8B5CF6 100%)',
    duration: '12 недель',
    lessons: '36 уроков'
  },
  { 
    id: 'professional', 
    name: 'Профессиональный', 
    icon: 'icons/level-pro.svg',
    color: '#F59E0B',
    description: 'Мастерство для профессионалов',
    gradient: 'linear-gradient(135deg, #F59E0B 0%, #EC4899 100%)',
    duration: '16 недель',
    lessons: '48 уроков'
  },
  { 
    id: 'retouch', 
    name: 'Ретушь', 
    icon: 'icons/level-retouch.svg',
    color: '#8B5CF6',
    description: 'Обработка и ретушь фотографий',
    gradient: 'linear-gradient(135deg, #8B5CF6 0%, #F59E0B 100%)',
    duration: '6 недель',
    lessons: '18 уроков'
  }
]

const instructors = [
  { name: 'Анна Петрова', experience: '12 лет', rating: 5, specialty: 'Портретная съемка', students: '5000+' },
  { name: 'Михаил Соколов', experience: '15 лет', rating: 5, specialty: 'Пейзажная фотография', students: '7000+' },
  { name: 'Елена Волкова', experience: '10 лет', rating: 5, specialty: 'Студийная съемка', students: '4000+' },
  { name: 'Дмитрий Иванов', experience: '18 лет', rating: 5, specialty: 'Ретушь и обработка', students: '9000+' }
]

const benefits = [
  { icon: 'icons/benefit-award.svg', title: 'Сертификат', text: 'Официальный сертификат по окончании' },
  { icon: 'icons/benefit-clock.svg', title: 'Гибкий график', text: 'Учитесь в удобное время' },
  { icon: 'icons/benefit-users.svg', title: 'Сообщество', text: 'Общение с единомышленниками' },
  { icon: 'icons/benefit-trending.svg', title: 'Карьерный рост', text: 'Помощь в развитии карьеры' }
]

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadCourses()
  updateStats()
  
  setInterval(() => {
    loadCourses()
    updateStats()
  }, 500)
}

function loadCourses() {
  const saved = localStorage.getItem('photoCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    courses = [
      {
        id: 1,
        module: 'Основы композиции',
        instructor: 'Анна Петрова',
        completionDate: '2025-03-25',
        rating: 5
      },
      {
        id: 2,
        module: 'Работа со светом',
        instructor: 'Михаил Соколов',
        completionDate: '2025-03-20',
        rating: 5
      },
      {
        id: 3,
        module: 'Портретная съемка',
        instructor: 'Елена Волкова',
        completionDate: '2025-03-15',
        rating: 4
      }
    ]
    localStorage.setItem('photoCourses', JSON.stringify(courses))
  }
  
  const coursesGrid = document.getElementById('coursesGrid')
  if (coursesGrid) {
    if (courses.length === 0) {
      coursesGrid.innerHTML = `
        <div class="empty-state">
          <div class="empty-icon-circle">
            <img src="icons/logo-camera.svg" alt="Camera" width="80" height="80">
          </div>
          <h3 class="empty-title">Нет курсов</h3>
          <p class="empty-text">Запишитесь на курс прямо сейчас</p>
        </div>
      `
    } else {
      coursesGrid.innerHTML = courses.map(course => {
        const stars = Array.from({ length: 5 }, (_, i) => 
          `<span class="star">${i < course.rating ? '★' : '☆'}</span>`
        ).join('')
        
        const date = new Date(course.completionDate).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
        
        return `
          <div class="course-card">
            <div class="course-header">
              <div class="course-module-badge">${course.module}</div>
              <div class="course-rating">${stars}</div>
            </div>
            <div class="course-content">
              <div class="course-info-item">
                <img src="icons/course-instructor.svg" alt="Instructor" width="24" height="24">
                <span>${course.instructor}</span>
              </div>
              <div class="course-info-item">
                <img src="icons/course-date.svg" alt="Date" width="24" height="24">
                <span>Завершен: ${date}</span>
              </div>
            </div>
          </div>
        `
      }).join('')
    }
  }
}

function updateStats() {
  const saved = localStorage.getItem('photoCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  }
  
  const totalCourses = courses.length
  const uniqueInstructors = new Set(courses.map(c => c.instructor)).size
  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0
  
  const stats = [
    {
      value: totalCourses,
      label: 'Курсов',
      icon: 'icons/stat-courses.svg',
      color: 'rgba(139, 92, 246, 0.2)'
    },
    {
      value: averageRating,
      label: 'Средняя оценка',
      icon: 'icons/stat-rating.svg',
      color: 'rgba(236, 72, 153, 0.2)'
    },
    {
      value: uniqueInstructors,
      label: 'Преподавателей',
      icon: 'icons/stat-instructors.svg',
      color: 'rgba(245, 158, 11, 0.2)'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color};">
          <img src="${stat.icon}" alt="" width="48" height="48">
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
}

function initEnrollPage() {
  const enrollmentForm = document.getElementById('enrollmentForm')
  const levelsGrid = document.getElementById('levelsGrid')
  const dateInput = document.getElementById('startDate')
  const successMessage = document.getElementById('successMessage')
  const benefitsGrid = document.getElementById('benefitsGrid')
  const instructorsGrid = document.getElementById('instructorsGrid')

  let selectedLevel = null

  if (levelsGrid) {
    levelsGrid.innerHTML = courseLevels.map(level => `
      <button type="button" class="level-card" data-id="${level.id}" style="border-color: rgba(139, 92, 246, 0.25);">
        <div class="level-icon-wrapper" style="background: ${level.color}30;">
          <img src="${level.icon}" alt="${level.name}" width="52" height="52">
        </div>
        <div class="level-content">
          <h3 class="level-name">${level.name}</h3>
          <p class="level-description">${level.description}</p>
          <div class="level-meta">
            <span class="level-duration">
              <img src="icons/benefit-clock.svg" alt="Clock" width="18" height="18">
              ${level.duration}
            </span>
            <span class="level-lessons">
              <img src="icons/benefit-award.svg" alt="Book" width="18" height="18">
              ${level.lessons}
            </span>
          </div>
        </div>
      </button>
    `).join('')

    levelsGrid.querySelectorAll('.level-card').forEach(button => {
      button.addEventListener('click', function() {
        levelsGrid.querySelectorAll('.level-card').forEach(btn => {
          btn.classList.remove('selected')
          btn.style.borderColor = 'rgba(139, 92, 246, 0.25)'
          const check = btn.querySelector('.level-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        this.style.borderColor = courseLevels.find(l => l.id === this.dataset.id).color
        selectedLevel = courseLevels.find(l => l.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'level-check'
        check.style.background = selectedLevel.gradient
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="32" height="32">`
        this.appendChild(check)
      })
    })
  }

  if (benefitsGrid) {
    benefitsGrid.innerHTML = benefits.map(benefit => `
      <div class="benefit-card">
        <div class="benefit-icon">
          <img src="${benefit.icon}" alt="" width="44" height="44">
        </div>
        <h3 class="benefit-title">${benefit.title}</h3>
        <p class="benefit-text">${benefit.text}</p>
      </div>
    `).join('')
  }

  if (instructorsGrid) {
    instructorsGrid.innerHTML = instructors.map(instructor => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="instructor-card">
          <div class="instructor-avatar">
            <img src="icons/instructor-avatar.svg" alt="" width="44" height="44">
          </div>
          <h3 class="instructor-name">${instructor.name}</h3>
          <p class="instructor-experience">${instructor.experience} опыта</p>
          <div class="instructor-rating">
            <span class="star">${stars}</span>
          </div>
          <p class="instructor-specialty">${instructor.specialty}</p>
          <div class="instructor-students">
            <img src="icons/benefit-users.svg" alt="" width="18" height="18">
            <span>${instructor.students} студентов</span>
          </div>
        </div>
      `
    }).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (enrollmentForm) {
    enrollmentForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedLevel && dateInput.value) {
        const savedCourses = localStorage.getItem('photoCourses')
        const courses = savedCourses ? JSON.parse(savedCourses) : []
        
        const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
        const weeks = parseInt(selectedLevel.duration)
        const completionDate = new Date(new Date(dateInput.value).getTime() + weeks * 7 * 24 * 60 * 60 * 1000)
        
        const newCourse = {
          id: Date.now(),
          module: selectedLevel.name,
          instructor: randomInstructor.name,
          completionDate: completionDate.toISOString().split('T')[0],
          rating: 0
        }

        courses.push(newCourse)
        localStorage.setItem('photoCourses', JSON.stringify(courses))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedLevel = null
          dateInput.value = ''
          levelsGrid.querySelectorAll('.level-card').forEach(btn => {
            btn.classList.remove('selected')
            btn.style.borderColor = 'rgba(139, 92, 246, 0.25)'
            const check = btn.querySelector('.level-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  if (slides.length === 0) return
  
  slides[currentSlide].classList.remove('active')
  if (dots[currentSlide]) dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  if (dots[currentSlide]) dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

